-------------------------------------------------------------------------------------------
--   File:			MergeAnimation.MS
--   Description:	A utility for merging complex animation between scenes
--   By:			Ravi Karra [Discreet] 			ravi.karra@autodesk.com
--				    Randy Kreitzman [Discreet]		randy.kreitzman@autodesk.com
--   Created:		1/26/01
--	Modified:		11/02/02
/*
	7/16/01		Added support for CA's
	7/31/01		Added auto name mapping button
				isAnimated state now checks for custom attributes
	11/05/02    Removed major chunk of old tree view related code
	08/11/03 	Atilla Szabo. Added code to delete anim keys on CUstom Attributes of base object and modifier. 
				CAs on the character head's and member's material should not be deleted by the Reset All Animation command
	4/2006      Chris Johnson. Removed activeX code, and replaced with .NET code.
	
*/
-------------------------------------------------------------------------------------------
  global gXmlIO, rObjectMapping, rMergeAnim, gEnableDebug 
  
  global lvops  --The ListViewOps struct definition found in <3dsmax>\stdplugs\stdscripts\NET_ListViewWrapper.ms
 
  rollout rObjectMapping ~ROBJECTMAPPING_CAPTION~ width:~ROBJECTMAPPING_WIDTH~ height:520
  (
	local cur_width = 336, src_width = 230, sry = 0, merge_col = 0, enable_debug = if (gEnableDebug == undefined) then false else gEnableDebug

	-- Node level drag & drop variables
	local in_drop = false, drag_node, drop_node
	
	label 		lblCurrent	~LBLCURRENT_CAPTION~	pos:[src_width+25, 2]
	label 		lblSource	~LBLSOURCE_CAPTION~	pos:[05, 2]	
	editText	etCurrent	"" 			pos:[(src_width+21), 015] width:(cur_width-16)	height:17
	button 		btnCurUpdate"U"			pos:[(cur_width+src_width+4), 015] width:20 height:17 iconName:"MergeAnimation/UpdateList" iconSize:[16,16] tooltip:~BTNCURUPDATE_TOOLTIP~
	editText 	etSource 	"" 			pos:[001, 015] width:(src_width-18) height:17
	button 		btnSrcUpdate"U"			pos:[(src_width-15), 015] width:20 height:17 iconName:"MergeAnimation/UpdateList" iconSize:[16,16] tooltip:~BTNSRCUPDATE_TOOLTIP~

	button 		btnUp		"^"			pos:[src_width+5, 100] width:20 iconName:"Common/ArrowUp" iconSize:[16,16] tooltip:~BTNUP_TOOLTIP~
	button 		btnDel		"x"			pos:[src_width+5, 130] width:20 iconName:"MergeAnimation/ClearSelected" iconSize:[16,16] tooltip:~BTNDEL_TOOLTIP~
	button 		btnDn		"v"			pos:[src_width+5, 160] width:20 iconName:"Common/ArrowDown" iconSize:[16,16] tooltip:~BTNDN_TOOLTIP~
	button		btnAutoMap	"A"			pos:[src_width+5, 190] width:20 iconName:"MergeAnimation/AutoNameMapping" iconSize:[16,16] tooltip:~BTNAUTOMAP_TOOLTIP~

	---------------------------------------------------------------------------------------
	groupBox 	grpDisplay	~GRPDISPLAY_CAPTION~ 	pos:~GRPDISPLAY_POSITION~ width:~GRPDISPLAY_WIDTH~ height:40
	---------------------------------------------------------------------------------------
	checkbox 	chkOnlyAnim ~CHKONLYANIM_CAPTION~	pos:~CHKONLYANIM_POSITION~ width:~CHKONLYANIM_WIDTH~ checked:false
	spinner		spnIndent 	~SPNINDENT_CAPTION~ 				pos:~SPNINDENT_POSITION~ type:#integer range:[0,20,1] width:50

	button	 btnLoadMapping	~BTNLOADMAPPING_CAPTION~ 	pos:~BTNLOADMAPPING_POSITION~ width:~BTNLOADMAPPING_WIDTH~ height:20
	button	 btnSaveMapping	~BTNSAVEMAPPING_CAPTION~ 	pos:~BTNSAVEMAPPING_POSITION~ width:~BTNSAVEMAPPING_WIDTH~ height:20
	
	local animDir = pathConfig.getDir #animations + "\\"
     local loadSeed = animDir

	dotNetControl lvSource  "ListView" pos:[005, 040]          width: src_width			height: 310	-- source objects listview
	dotNetControl lvCurrent "ListView" pos:[src_width+25, 040] width:(src_width + 106) 	height: 310	-- current objects listview		  			
	
	function sortList lv key =
	(
		setWaitCursor()
		/*	
		lv.sortKey = key
		local order = lv.sortOrder
		lv.sortOrder = if order == #lvwAscending then #lvwDescending else #lvwAscending
		lv.sorted = true  -- sort for now
		lv.sorted = false -- disabling auto sorting
		*/
		setArrowCursor()
	)

	function doRollup state = 
	(
		if state then sry = 400 else sry = 0 
		
		rMergeAnim.height = 250 + sry
		rMergeAnim.srRollouts.height = sry + 30
		--format "% % - % - %\n" num_rolls state sry rMergeAnim.srRollouts.height
		rMergeAnim.statusBar.pos = [5, sry + 225]
		rMergeAnim.pbStatus.pos = [360, sry + 226]		
		rMergeAnim.statusBar.width = 350
		rMergeAnim.statusBar.height= 20
	)
	
	on rObjectMapping rolledUp state do
	(
		doRollup state
	)
	
	on rObjectMapping open do --fixed
	(
		-- Initialize the ListView controls
		local l_widthA = (rObjectMapping.cur_width/2 - 2)
		local l_widthB = (rObjectMapping.src_width - 4)
		lvops.InitListView lvCurrent pInitColumns:#(~CURRENT_NODES~, ~MERGE_NODES~) pAllowDrop: true pInitColWidths:#(l_widthA,l_widthA ) pLabelEdit:false
		lvops.InitListView lvSource  pInitColumns:#(~SOURCE_NODES~)                 pAllowDrop: true pInitColWidths:#(l_widthB)          pLabelEdit:false
		
		lvCurrent.hideSelection = false
		lvSource.hideSelection = false
		
		lvops.InitImageList lvSource #("$maxSysIcons/tvObj.ico") 
		lvops.InitImageList lvCurrent #("$maxSysIcons/tvObj.ico") pSize: 10
	)
	
	on etCurrent entered val do rMergeAnim.updateCurrent()
	on etSource entered val do rMergeAnim.updateSource()
	on btnCurUpdate pressed do rMergeAnim.updateCurrent()
	on btnSrcUpdate pressed do rMergeAnim.updateSource()
	on chkOnlyAnim changed val do ( rMergeAnim.updateSource() )
		
	function IsSourceMergedAndMapped lvItemArray =
	(
		local mergedList  = #()
		for lvitem in lvItemArray do
		(
			if (lvitem.subitems.count >= 2) do
			(
				append mergedList true
			)
		)
		if (mergedList.count > 0) then true else false
	)
	on btnUp pressed do
	(
		local selectedLVItems = lvops.GetLvSelection lvCurrent
		local lvitems = lvops.GetLvItems lvCurrent
		--Only proceed if something is selected
		if (selectedLVItems.count > 0) then
		(
			--Only proceed if a source has been opened, and at least one
			--one object has been mapped
			if (IsSourceMergedAndMapped lvitems) do
			(
				if lvitems.count == 0 or lvitems[1].selected then  ( /* return */ )
				else ( rMergeAnim.moveLVItems lvitems 2 lvitems.count 1 )
			)
		)
	)	
	on btnDn pressed do
	(
		local selectedLVItems = lvops.GetLvSelection lvCurrent
		local lvitems = lvops.GetLvItems lvCurrent
		--Only proceed if something is selected
		if (selectedLVItems.count > 0) then
		(
			--Only proceed if a source has been opened, and at least one
			--one object has been mapped
			if (IsSourceMergedAndMapped lvitems) do
			(
				--i.e. Bail when the moved item is now at the bottom.
				if lvitems.count == 0  or lvitems[lvitems.count].selected then ( /* return() */ ) 
				else ( rMergeAnim.moveLVItems lvitems (lvitems.count - 1) 1 -1 )
			)
		)
	)
	on btnDel pressed do rMergeAnim.clearSelected lvCurrent
	on btnAutoMap pressed do --fixed
	(
		setWaitCursor()
		rMergeAnim.setStatus ~RMERGEANIM_SETSTATUS~
		local sidx = findString etSource.text "*"
		local cidx = findString etCurrent.text "*"
		item_count = lvops.GetLvItemCount lvSource
		local progressindex = 1

		-- first gather objects from the left window
		local LvItems = lvops.GetLvItems lvSource

		--For each listview item in left window 
		for lis in LvItems do 
		(
			local l_text = lis.text

			-- form the search string from prefix of current + suffix of source		
			if sidx != undefined then
			(
				l_text = subString etCurrent.text 1 (cidx-1) + subString lis.text sidx (lis.text.count - sidx + 1)
			)
			
			--And query for a a match of a list item with the same name in the 
			-- right window. .NET stores the result in an array
			local lv_nodes = lvops.GetLvItemByname lvCurrent l_text
			--If the array is valid, then continue
			if (lv_nodes != undefined) and (lv_nodes.count > 0) then
			(
				local lv_node = lv_nodes[1]
				if lv_node != undefined then 
				(
					-- set the text and tag properties of the Listview's subitem
					local mergeSubItem = undefined
					--if there is no items in the second column, then create a new one
					if (lv_node.subItems.count == 1) then
					(	
						--mergeSubItem = lv_node.subItems.add l_text
						--throw "This should NOT happen"
					)
					--otherwise use the existing one.
					else if (lv_node.subItems.count == 2) then
					(
						mergeSubItem = lv_node.subItems.item[1]
						mergeSubItem.text = lis.text
					)
					-- copy the tv node tag to the list item tag
					mergeSubItem.tag = lis.tag
				)
			)
			rMergeAnim.pbStatus.value = (100.0 * progressindex)/item_count
			progressindex += 1
		)

		lvops.RefreshListView lvCurrent
		rMergeAnim.pbStatus.value = 0

		rMergeAnim.setStatus ~RMERGEANIM_SETSTATUS_AUTO_MAPPING_COMPLETED~
	)
	on btnLoadMapping pressed do
	( 
		local f = getOpenFileName filename:loadSeed caption:~OPEN_MERGE_ANIMATION_MAPPING_CAPTION~ types:~MAM_FILES_TYPES~
		if f != undefined then
		(
		    loadSeed = f
			rMergeAnim.map_stream = openFile f
			local res = rMergeAnim.loadMapping()
			close rMergeAnim.map_stream
			rMergeAnim.setStatus (if res then ~MAPPING_FILE_LOADED~ else ~MAPPING_FILE_LOAD_FAILED~)
		)
	)
	on btnSaveMapping pressed do 
	( 
		local f = getSaveFileName filename:animDir caption:~SAVE_MERGE_ANIMATION_MAPPING~ types:~MERGE_MAM_FILES_TYPES~
		if f != undefined then
		(
		    animDir = f
			rMergeAnim.map_stream = createFile f
			rMergeAnim.saveMapping()
			close rMergeAnim.map_stream
		)		
--		print map_stream --For debugging
	)
	
	
	on lvCurrent ItemClick lvn do
	(
		lvn = lvCurrent.selectedItem
		if lvn != undefined then
		(
			lvCurrent.dropHighlight = undefined
			local si = lvn.SubItems.item[merge_col]
			rMergeAnim.setStatus (if si == undefined or rMergeAnim.xml_input or (getNodeByName si.text) ==  undefined then "" else (~ANIMATION_FROM~ + si.tag.name))
		)
	)
	on lvCurrent columnClick columnHeader do ()--sortList lvCurrent (columnHeader.index - 1)
	on lvSource columnClick columnHeader do ()--sortList lvSource (columnHeader.index - 1)
	on lvSource ItemClick lvn do
	(
		lvn = lvSource.selectedItem
		if lvn != undefined then
		(
			rMergeAnim.setStatus (if rMergeAnim.xml_input then (rMergeAnim.xmlIO.getAttribute (rMergeAnim.xmlIO.xmlDoc.selectSingleNode ("//object[@id='" + lvn.tag + "']")) "name") else lvn.tag.name)
		)
	)

	-------------------------------------------------------------------------------------------
	-- All the drag & drop stuff
	-------------------------------------------------------------------------------------------
--	on lvSource DragEnter arg do
--	(
--		-- arg is System.Windows.Forms.DragEventArgs
--		if (arg.data.GetDataPresent(dotNetClass "System.Windows.Forms.ListViewItem")) do
--		(
--			local dragDropEffect = dotNetclass "System.Windows.Forms.DragDropEffects"
--			arg.effect = dragDropEffect.move
--		)
--		in_drop = false
--	)
	on lvSource ItemDrag arg do
	(
		-- arg is System.Windows.Forms.ItemDragEventArgs
		drag_node = arg.item
		local effect = dotNetclass "System.Windows.Forms.DragDropEffects"	
		lvSource.DoDragDrop arg.item effect.move
	)
	on lvCurrent DragEnter arg do
	(
		-- arg is System.Windows.Forms.DragEventArgs
		if (arg.data.GetDataPresent(dotNetClass "System.Windows.Forms.ListViewItem")) do
		(
			local dragDropEffect = dotNetclass "System.Windows.Forms.DragDropEffects"
			arg.effect = dragDropEffect.move
			in_drop = true
			lvCurrent.multiselect = false
		)
	)
	on lvCurrent DragLeave arg do
	(
		-- arg is System.EventArgs
		if (in_drop) do
		(
			in_drop = false
			lvCurrent.multiselect = true
		)	
	)
	
	on lvCurrent DragOver arg do
	(
		--This highlights the ListViewItem that is directly under the mouse
   		--arg is System.Windows.Forms.DragEventArgs
   		if (arg.data.GetDataPresent(dotNetClass "System.Windows.Forms.ListViewItem")) then
		(
			local position = dotNetObject "System.Drawing.Point" arg.x arg.y
			position         = lvCurrent.pointToClient position
			local hitTestInfo = lvCurrent.HitTest position
			local dropLVItem = hitTestInfo.Item

			if dropLVItem != undefined then
			(
				dropLVItem.selected = true
			)
		)
	)
	
	on lvCurrent DragDrop arg do 
	( 			
		-- arg is System.Windows.Forms.DragEventArgs
		if (arg.data.GetDataPresent(dotNetClass "System.Windows.Forms.ListViewItem")) do
		(
			local dnPointCls    = dotNetObject "System.Drawing.Point" arg.x arg.y
			local position      = lvCurrent.pointToClient dnPointCls
			local hitTestInfo   = lvCurrent.HitTest position
			local dropIndex     = hitTestInfo.Item.index
			
			local selectedSourceItems = lvops.GetLvSelection lvSource
			local selectedCount       = selectedSourceItems.count
			local lvCurrentItems      = lvops.GetLvItems lvCurrent
			
			local i = 1
			--for each selected dragged node
			local notDone = true
			for dragged in selectedSourceItems while notDone do
			(
				--get a scene node to assign with.
				lvItem = lvCurrentItems[dropIndex + i]
				--and if the end of the list hasn't been reached
				if (lvItem != undefined) then 
				(
					--then get the subitem in the second column.
					lvSubItem = lvItem.subItems.item[1]
					
					--now copy the actual data
					lvsubitem.text = dragged.text
					lvsubitem.tag  = dragged.tag
					
					--increment the counter
					i += 1
				)
				else
				(
					--most likely reached the end of the list. 
					notDone = false
				)
			)
			lvCurrent.multiselect = true
			gc()
		)
	)

	
	on lvCurrent DoubleClick arg do
	( 
		lvItem = lvops.GetLvSingleSelected lvCurrent
		if lvItem != undefined and lvItem.tag != undefined then select lvItem.tag.value
	)
	on lvCurrent keyUp arg do
	(	
		-- arg is System.Windows.Forms.KeyEventArgs
		-- ASCII value for SPACEBAR, BACKSPACE and DELETE
		local key = arg.keyValue
		if ( key == 32 or key == 8 or key == 46) do
		(	
			rMergeAnim.clearSelected lvCurrent
		)
	)
  ) --end of rollout
  
  
  rollout rMergeAnim ~ROLLOUT_MERGE_ANIMATION~ width:620
  (
     local sourceSeed = pathConfig.getDir #animations + "\\"
	local xmlIO, dontFilterChildren = false, current_range = interval 0 0
	local	merge_col = 1, xml_input = false, 
			ini_file = ini_file = ((getDir #plugcfg) + "\\mergeAnim.ini"),
			is_holding = false, hold_objects = #(), hold_file = ((getDir #autoback) + "\\__merge_anim.mx") -- hold related
			
	local file_name = undefined, new_xref, new_tree, current_nodes, source_nodes, src_node, whenHandle, mergAnimID = 0x1ef782c5, map_stream, replaceController_fn
	
	local show_messages = true
	
	-- various colors
	local anim_color = color 255 0 0
	local proc_color = color 255 255 0
	local err_color  = color 0 255 0

	-- Sub-anim level drag & drop variables
	local in_sub_drop = false, sub_drag_node, sub_drop_node
	
	-- Temporary vertical UI offset
	local offset = -140, offset2 = 51, x_offset = ~X_OFFSET~, my = -70, aty = -210, ax = ~AX_VALUE~, ay = 100, sy = 5
	
	-- Merge variables
	local mergeNodeArray = #(), new_tree_count = 0

	-- Procedural types
	local proc_types = #(
			position_script,
			rotation_script,
			scale_script,
			float_script,
			position_expression,
			scale_expression,
			float_expression,
			position_wire,
			rotation_wire,
			scale_wire,
			float_wire	
			)
			
	-- counters
	local savedNodes=0, loadedNodes=0, obj_count = 1

	---------------------------------------------------------------------------------------
	groupBox grpFile		~GRPFILE_CAPTION~	pos:[005,~GRPFILE_POSITION~] width:~GRPFILE_WIDTH~ height:~GRPFILE_HEIGHT~
	---------------------------------------------------------------------------------------	
	button	 btnSource		~BTNSOURCE_CAPTION~ 		pos:[010,sy+~BTNSOURCE_POSITION~] width:100 height:20
	button	 btnSourceObj	~BTNSOURCEOBJ_CAPTION~ 	pos:[120,sy+~BTNSOURCEOBJ_POSITION~] width:100 height:20	
	label    lblProps "" pos:[230,sy+~LBLPROPS_POSITION~] width:~LBLPROPS_WIDTH~ height:~LBLPROPS_HEIGHT~		-- file properties

	button 	 btnSave2XML	~BTNSAVE2XML_CAPTION~	pos:[ax+~BTNSAVE2XML_POSITION~,ay+015] width:~BTNSAVE2XML_WIDTH~ height:20
	button	 btnMerge		~BTNMERGE_CAPTION~	pos:[ax+~BTNMERGE_POSITION~,ay+045] width:~BTNMERGE_WIDTH~ height:20
	button	 btnFetch		~BTNFETCH_CAPTION~	pos:[ax+~BTNFETCH_POSITION~,ay+075] width:~BTNFETCH_WIDTH~ height:20 enabled:false
		
	---------------------------------------------------------------------------------------
	groupBox grpSource		~GRPSOURCE_CAPTION~ pos:[005,125+my] width:~GRPSOURCE_WIDTH~ height:~GRPSOURCE_HEIGHT~
	---------------------------------------------------------------------------------------
	radiobuttons rb_ctrl						pos:[010,140+my] labels:#(~REPLACE_ANIMATION~, ~PASTE_TO_EXISTING_ANIMATION~) columns:1 default:1
	
	checkbox cb_matchRange	~MATCH_SOURCE_FILE_TIME~ pos:[~CB_MATCHRANGE_POSITION~,175+my] width:~CB_MATCHRANGE_WIDTH~ highlightColor:green enabled:false
	spinner	s_startTime		~S_STARTTIME_CAPTION~ 		pos:[20,195+my] type:#integer fieldwidth:35 range:[-100000,100000,animationRange.start] enabled:false
	spinner	s_endTime		~S_ENDTIME_CAPTION~ 		pos:[130,195+my] type:#integer fieldwidth:35 range:[-100000,100000,animationRange.end] enabled:false

	spinner s_insert ~S_INSERT_CAPTION~ pos:[~S_INSERT_POSITION~,215+my] type:#integer fieldwidth:30 range:[-100000,100000,currentTime]
	radiobuttons rb_relAbs						pos:[15,~RB_RELABS_POSITION~+my] labels:#(~RELATIVE~, ~ABSOLUTE~) enabled:false columns:2

	checkbox chkAdjustTimeRange ~CHKADJUSTTIMERANGE_CAPTION~ pos:[15,245+my] width:~CHKADJUSTTIMERANGE_WIDTH~ height:~CHKADJUSTTIMERANGE_HEIGHT~ highlightColor:(color 0 255 0) enabled:true
	
	---------------------------------------------------------------------------------------
	groupBox grpApply 		~GRPAPPLY_CAPTION~ 		pos:[x_offset,265 +aty] width:~GRPAPPLY_WIDTH~ height:~GRPAPPLY_HEIGHT~
	---------------------------------------------------------------------------------------

	---------------------------------------------------------------------------------------
	groupBox grpMainAttribs ~GRPMAINATTRIBS_CAPTION~ pos:[x_offset+5,(~GRPMAINATTRIBS_POSITION~ + aty)] width:~GRPMAINATTRIBS_WIDTH~ height:~GRPMAINATTRIBS_HEIGHT~
	---------------------------------------------------------------------------------------
	checkbox chkTransform 	~CHKTRANSFORM_CAPTION~ 	pos:[x_offset+10,(300 + aty)] height:015 checked:true
	checkbox chkIK			"IK"			pos:[x_offset+~CHKIK_POSITION~,(300 + aty)] width:30 height:015 checked:true
	checkbox chkPosition 	~CHKPOSITION_CAPTION~ 		pos:[x_offset+20,(320 + aty)] width:80 height:015 checked:true
	checkbox chkRotation 	~CHKROTATION_CAPTION~ 		pos:[x_offset+20,(340 + aty)] width:80 height:015 checked:true
	checkbox chkScale 		~CHKSCALE_CAPTION~ 		pos:[x_offset+20,(360 + aty)] width:80 height:015 checked:true
	checkbox chkModifiers 	~CHKMODIFIERS_CAPTION~ 	pos:[x_offset+10,(380 + aty)] width:~CHKMODIFIERS_WIDTH~ height:015 checked:true
	
	---------------------------------------------------------------------------------------
	groupBox grpMoreAttribs ~GRPMOREATTRIBS_CAPTION~ pos:[x_offset+~X_CONSTANT_OFFSET~,(~Y_CONSTANT_OFFSET~ + aty)] width:~GRPMOREATTRIBS_WIDTH~ height:~GRPMOREATTRIBS_HEIGHT~
	---------------------------------------------------------------------------------------
	checkbox chkCustAttrib 	~CHKCUSTATTRIB_CAPTION~ pos:[x_offset+~CHKCUSTATTRIB_X_OFFSET~,(300 + aty)] width:~CHKCUSTATTRIB_WIDTH~ height:015 checked:true
	checkbox chkAddNewDefs	~CHKADDNEWDEFS_CAPTION~ 	pos:[x_offset+~X_OFFSET_CONSTANT~,(320 + aty)] width:~CHKADDNEWDEFS_WIDTH~ height:015 checked:true
	checkbox chkBaseObject 	~CHKBASEOBJECT_CAPTION~ 	pos:[x_offset+~CONST_OFFSET_X~,(340 + aty)] width:100 height:015 checked:false
	checkbox chkMaterials 	~CHKMATERIALS_CAPTION~ pos:[x_offset+~OFFSET_X_CONSTANT~,(360 + aty)] width:100 height:015 checked:false
	checkbox chkVisTracks 	~CHKVISTRACKS_CAPTION~ pos:[x_offset+~X_OFFSET_CONST_VALUE~,(380 + aty)] width:~CHKVISTRACKS_WIDTH~ height:015 checked:false


	subRollout srRollouts	""				pos:[005,200] width: 610 rolledUp: true
	editText   statusBar	"" 				width: 350 height: 20 enabled: false
	progressBar  pbStatus	""				width:255 height:20 color:blue
	---------------------------------------------------------------------------------------
	-- Functions
	---------------------------------------------------------------------------------------
	function hasProcedural node =
	(
		local tm = node[3].controller
		if tm == undefined then return false
		if (classof tm.controller) == transform_script then return true
		for i=1 to tm.numSubs do
		(
			local sa = getSubAnim tm i
--			format " %->% - %\n" node.name (getSubAnimName tm i) (classof sa.controller) as string
			if sa != undefined and ((findItem proc_types (classof sa.controller)) != 0) then
			(
				return true
			)
		)
		false
	)
	
	function hasXMLProcedural xmlNode =
	(
		local tm = xmlNode.selectSingleNode (#transform as string) 
		if tm == undefined then return false
		if (xmlIO.getAttribute tm #classOf) == "transform_script" then return true
		for c in tm.childNodes do
		(
			if (findItem class_types (xmlIO.getAttribute c #classOf)) != 0 then
				return true
		)
		false
	)	
	
	function getNodeByHandle handle = 
	(
		for o in objects do if o.handle == handle then return o
		undefined
	)
	-------------------------------------------------------------------------------------------
	-- Statusbar functions
	-------------------------------------------------------------------------------------------
	function initStatusBar = 
	(
		statusBar.enabled = false
	)
	function setStatus text error:false =
	(
		statusBar.text = (if error then ~STATUSBAR_IF_ERROR_CAPTION~ else "") + text
	)
	-------------------------------------------------------------------------------------------
	-- Listview functions
	-------------------------------------------------------------------------------------------	
	function clearSelected lv = 
	(
		local selLVItems = lvops.GetLvSelection lv
		for lvItem in selLVItems do
		(
			if (lvItem.subItems.count >= 2) do
			(
				local si = lvItem.subItems.Item[merge_col]
				si.text = ""
				si.tag = undefined
			)
		)
	)
	function swapItems li1 li2 = 
	(
		local text1, text2, tag1, sel1 = li1.selected
		text1 = li1.subItems.item[merge_col].text
		text2 = li2.subItems.item[merge_col].text
		tag1  = if text1 != "" then li1.SubItems.item[merge_col].tag else ""
		
		li1.selected = li2.selected
		li2.selected = sel1
		
		if text2 == "" then
		(
			--li1.SubItems.removeat 1
			li1.SubItems.item[merge_col].text = ""
			li1.SubItems.item[merge_col].tag = undefined
			
		)
		else
		(
			li1.subItems.item[merge_col].text = text2
			li1.SubItems.item[merge_col].tag = li2.SubItems.item[merge_col].tag			
		)
		if text1 == "" then
		(
			--li2.SubItems.removeat 1
			li2.SubItems.item[merge_col].text = ""
			li2.SubItems.item[merge_col].tag = undefined
		)
		else
		(
			li2.subItems.item[merge_col].text = text1
			li2.SubItems.item[merge_col].tag = tag1
		)
	)
	function moveLvItems lvItems start last byNum =
	(
		setWaitCursor()
		for i = start to last by byNum do
		(
			if not lvItems[i-byNum].selected and lvItems[i].selected do
			(
			 	start = i
				exit
			)			
		)
		for i = start to last by byNum do
		(
			local si = lvitems[i].subItems.item[merge_col]
			if (si != undefined) do
			(
				if lvitems[i].selected and si != undefined and si.text != "" then
				(
					swapItems lvitems[i] lvitems[i-byNum]
				)
			)
		)
		setArrowCursor()
	)
	
	function addSceneObjectsToListView lv max_nodes patternstring:undefined recurse:true indent:0 =
	(
		local lv_nodes = lvops.GetLvItems lv
			
		local selectedCount = max_nodes.count 
		for i = 1 to selectedCount do
		(
			local c = max_nodes[i]
			local animated = xmlio.isAnimated c
			local add_node = (c.name != "_merge_anim")
			
			-- do node filtering based upon the pattern string
			if add_node and patternstring != undefined and patternstring != "" then
			(
				add_node = if patternstring == "$" then c.isSelected else matchPattern c.name pattern:patternstring
			)
				
			-- do node filtering based "Show Animated" checkbox status
			if add_node and rObjectMapping.chkOnlyAnim.checked then
				add_node = animated
			
			--Add the node
			if add_node then
			(
				local lvn = lvops.AddLvItem lv pTextItems:#(c.name,"") pTag:(dotNetMXSValue c) pIndent:indent
				
				if (hasProcedural c) then
				(
					lvn.UseItemStyleForSubItems = false
					lvops.HighLightLvItem lvn.subitems.item[0] #bold proc_color
				)
				else if animated and not rObjectMapping.chkOnlyAnim.checked do
				(
					lvn.UseItemStyleForSubItems = false
					lvops.HighLightLvItem lvn.subitems.item[0]  #bold anim_color
				)				
			)
			
			--Recursively add any children to the listview
			if recurse and c.name != "_merge_anim" do
			(
				addSceneObjectsToListView lv c.children \
										patternstring: (if dontFilterChildren then undefined else patternstring) \
										recurse:       recurse \
										indent:        (indent + rObjectMapping.spnIndent.value)
			)
		)
	)
	
	function addXMLNodesToListView lv xml_nodes patternstring:undefined recurse:true indent:0 =
	(
		local lv_nodes = lv.Items.item
		for i=0 to (xml_nodes.length-1) do
		(
			local c = xml_nodes[i], animated = ((xmlIO.getAttribute c "isAnimated")=="true"), add_node = true, lvn
			-- do node filtering based upon the pattern string
			if add_node and patternstring != undefined and patternstring != "" then
			(
				add_node = if patternstring == "$" then c.isSelected else matchPattern (xmlIO.getAttribute c "name") pattern:patternstring
			)
				
			-- do node filtering based "Show Animated" checkbox status
			if add_node and rObjectMapping.chkOnlyAnim.checked then
				add_node = animated
			
			if add_node then
			(
				lvn = lv_nodes.add()
				lvn.text = (xmlIO.getAttribute c "name")
				listView.setIndent lv.hwnd lvn.index indent
				lvn.tag = xmlIO.getAttribute c "id" -- save the node handle in the tag property
				
				if (hasXMLProcedural c) then
				(
					lvn.UseItemStyleForSubItems = false
					lvops.HighLightLvItem lvn.subitems.item[0] #bold proc_color
				)
				else if animated and not rObjectMapping.chkOnlyAnim.checked do
				(
					lvn.UseItemStyleForSubItems = false
					lvops.HighLightLvItem lvn.subitems.item[0] #bold anim_color
				)				
			)
			if recurse then
				addXMLNodesToListView lv (c.selectNodes "children/object") patternstring:(if dontFilterChildren then undefined else patternstring) recurse:recurse indent:(indent + rObjectMapping.spnIndent.value)
		)
	)		

	-------------------------------------------------------------------------------------------
	-- Update functions
	-------------------------------------------------------------------------------------------	
	function updateCurrent = 
	(
		local item_count = lvops.GetLvItemCount rObjectMapping.lvCurrent
		if item_count != 0 do 
		(
			local text = rObjectMapping.etCurrent.text
			setWaitCursor()
			lvops.ClearLvItems rObjectMapping.lvCurrent
			
			current_nodes = if text == "$" then selection else rootNode.children
			local bRecurse = (if text == "$" then false else true) 
			addSceneObjectsToListView (rObjectMapping.lvCurrent) current_nodes patternstring:text recurse:bRecurse
			setArrowCursor()
		)
	)
	function updateSource = 	
	(
		--format "updateSource: %\n" source_nodes 
		if ((lvops.GetLvItemCount rObjectMapping.lvSource) == 0) do
		(	
			local text = rObjectMapping.etSource.text
			if new_tree != undefined do
			(
				setWaitCursor()
				lvops.ClearLvItems rObjectMapping.lvSource

				if xml_input then
					addXMLNodesToListView (rObjectMapping.lvSource) source_nodes patternstring:(if text == "$" then undefined else text) recurse:true
				else
					addSceneObjectsToListView (rObjectMapping.lvSource) source_nodes patternstring:(if text == "$" then undefined else text) recurse:true
				setArrowCursor()
			)
		)
	)	
	
	function updateCurrentList reload:true =
	(
		local srcSel, curSel
		try if rObjectMapping.lvCurrent.selectedItem != undefined do curSel = rObjectMapping.lvCurrent.selectedItem.index
		catch()
		if reload then
			rMergeAnim.current_nodes = rootNode.children
		updateCurrent()
		try if curSel != undefined do rObjectMapping.lvCurrent.selectedItem = rObjectMapping.lvCurrent.nodes[curSel]
		catch()		
	)	
	-------------------------------------------------------------------------------------------
	-- Animation functions
	-------------------------------------------------------------------------------------------
	-- getRefTarget() recursively tracks down dependencies in a copied controller's reference hierarchy !!CURRENTLY NOT IN USE!!
	function getRefTarget fromNodeCtrl nodeArray =
	(
		for i = 1 to fromNodeCtrl.numsubs do
		(
			if fromNodeCtrl[i].controller != undefined do
			(
				getRefTarget fromNodeCtrl[i].controller nodeArray
			)
		)
		-- Collect fromNodeCtrl root node reference for later comparison
		local nodeArray2 = #()
		for t in (refs.dependents fromNodeCtrl) do
		(
			if (isKindOf t node) do
			(
				-- Hack to filter out classOf "node" objects
				try
				(
					t.name
					if (findItem nodeArray2 t) == 0 do append nodeArray2 t
				)
				catch()
			)
		)
		for r in (refs.dependsOn fromNodeCtrl) do
		(
			-- Depends on PB1/PB2 Reference Target (Path, Positon, Link constraint, etc...)
			if (findString (r as string) "ReferenceTarget:") != undefined do
			(
				local ref = refs.dependsOn r
				for t in ref do
				(
					if (isKindOf t node) then
					(
						if (findItem nodeArray t) == 0 do append nodeArray t
					)
					else if ref.count == 1 do r = t
				)
			)
			-- Depends on sub-anim controller (Param wires, instanced controllers, etc.)
			if (findString ((superClassOf r) as string) "Controller") != undefined do
			(
				local ref = refs.dependents r
				for t in ref do
 				(
					if (isKindOf t node) do
					(
						-- Hack to filter out classOf "node" objects
						try
						(
							t.name
							if (findItem nodeArray t) == 0 and (findItem nodeArray2 t) == 0 do append nodeArray t
						)
						catch()
					)
				)
			)
			-- Depends directly on node (Expression controller absolute node reference)
			if (isKindOf r node) do
			(
				-- Hack to filter out classOf "node" objects
				try
				(
					r.name
					if (findItem nodeArray r) == 0 do append nodeArray r
				)
				catch()
			)
		)
		nodeArray
	)
	
	function traverseSubAnims fromNodeCtrl toNodeCtrl =
	(
		if rObjectMapping.enable_debug do format "traverseSubAnims fromNodeCtrl:% toNodeCtrl:% toNodeCtrl.controller:%\n" fromNodeCtrl toNodeCtrl toNodeCtrl.controller
		for i = 1 to fromNodeCtrl.numsubs do
		(
			local sub_anim = getSubAnimName fromNodeCtrl i
			if rObjectMapping.enable_debug do format "%(%) -- %(%)\n" fromNodeCtrl[i] fromNodeCtrl[i].controller toNodeCtrl[sub_anim] toNodeCtrl[sub_anim].controller
			if toNodeCtrl.numsubs >= i and (sub_anim == getSubAnimName toNodeCtrl i) then
				sub_anim = i
			if rObjectMapping.enable_debug do format "  traverseSubAnims i:% sub_anim:% toNodeCtrl[sub_anim]:%\n" i sub_anim toNodeCtrl[sub_anim]
			if (toNodeCtrl[sub_anim] != undefined) then
			(
				if fromNodeCtrl[i].controller != undefined then 
					replaceController_fn fromNodeCtrl[i].controller &toNodeCtrl[sub_anim]
				else
					traverseSubAnims fromNodeCtrl[i] toNodeCtrl[sub_anim]
			)
		)
	)
	
	function createControllerHeirarchy fromNodeCtrl &toNodeCtrl =
	(
		toNodeCtrl.controller = createInstance (classof fromNodeCtrl) --execute (classOf fromNodeCtrl as string + "()")
		if ( list_ctrl = (findString (classOf toNodeCtrl.controller as string) "_list") != undefined ) then
		(
			-- also create the sub anim controllers
			for i=1 to (fromNodeCtrl.NumSubs-1) do -- -1 for skipping available slot
				createControllerHeirarchy fromNodeCtrl[i].controller &toNodeCtrl["available"]
		)
	)
	
	-- replaceController() copies/pastes all keys from the source controller to the destination controller, overwriting existing keys
	function replaceController fromNodeCtrl &toNodeCtrl =
	(	
		if rObjectMapping.enable_debug do format "replaceController fromNodeCtrl:% toNodeCtrl:% toNodeCtrl.controller:%\n" fromNodeCtrl toNodeCtrl toNodeCtrl.controller
		if fromNodeCtrl == undefined then return()
		
		-- Assign a new controller to a non-animated parameter, if necessary		
		try
		(
			if toNodeCtrl.controller == undefined or (classOf fromNodeCtrl) != (classOf toNodeCtrl.controller) do 
				createControllerHeirarchy fromNodeCtrl &toNodeCtrl
		) 
		catch( if rObjectMapping.enable_debug then format ~FORMATTED_MERGE_ERROR_CAPTION~ fromNodeCtrl toNodeCtrl)		
		
		if rObjectMapping.enable_debug do format "replaceController (classOf fromNodeCtrl):% toNodeCtrl.controller:% (classOf toNodeCtrl.controller):%\n" (classOf fromNodeCtrl) toNodeCtrl.controller (classOf toNodeCtrl.controller)
		-- Filter out incompatible controllers (such as Bezier_Position and Position_XYZ)
		if (classOf fromNodeCtrl) != (classOf toNodeCtrl.controller) do return()		
		
		-- Filter out Position/Euler XYZ controllers which return an invalid key array
		if (findItem #(Position_XYZ, Euler_XYZ, Master_Point_Controller) (classOf fromNodeCtrl)) == 0 do 
		(	
			try	(
				-- Filter out all but key-framed controllers
				if fromNodeCtrl.keys.count > 0 and toNodeCtrl.controller != fromNodeCtrl do with animate off 	(
					source_range = (getTimeRange fromNodeCtrl) -- DMW
					if source_range.start < current_range.start do current_range.start = source_range.start  -- DMW
					if source_range.end > current_range.end do current_range.end = source_range.end          -- DMW		
				
					deleteKeys toNodeCtrl.controller #allkeys
					for k in fromNodeCtrl.keys do
					(
						appendKey toNodeCtrl.controller.keys k
					)
				)
			) catch	( if rObjectMapping.enable_debug then format ~FORMATTED_CANNOT_ACCESS_KEYS_FOR_CONTROLLER~ fromNodeCtrl toNodeCtrl )
		)		
		-- Recursively traverse controller sub-anim tree looking for sub-controllers
		traverseSubAnims fromNodeCtrl toNodeCtrl
	)
	
	-- pasteToActiveCtrl() copies/pastes a range of keys (relative/absolute) from the source controller to the destination controller at an optional insertion point
	function pasteToActiveCtrl fromNodeCtrl &toNodeCtrl =
	(
		local 	source_range, toKeyArray = #(),
				insert_time = s_insert.value, 
				tempCtrl, toNodeVal = toNodeCtrl.value, deltaVal, fromCtrl, toCtrl,
				list_ctrl = (findString (classOf toNodeCtrl.controller as string) "_list") != undefined
		
		local deps = refs.dependents toNodeCtrl.controller		
		if fromNodeCtrl == undefined or ( findItem proc_types (classOf fromNodeCtrl) ) != 0 or (findItem proc_types (classOf toNodeCtrl.controller) ) != 0  do
		(
			if rObjectMapping.enable_debug then
				format  ~FORMAT_SKIPPING~ fromNodeCtrl toNodeCtrl.controller
			return()	
		)
		fromCtrl = fromNodeCtrl		
		
		-- If toNodeCtrl is a list controller then send the active controller downstream as toCtrl
		if list_ctrl and false then
		(
			if rObjectMapping.enable_debug then
				format  ~FORMAT_LIST_CONTROLLER~ fromCtrl toNodeCtrl.controller

			-- If the active controller is "Available" then create a new controller
			local listI = toNodeCtrl.controller.getActive()
			local activeAnim = toNodeCtrl.controller[listI] 
			if listI==0 and activeAnim == undefined then 
			(
				toNodeCtrl.available.controller = createInstance (classof fromCtrl)
				listI = 1
			)
			else if listI == 1 do 
			(
				local activeCtrl = activeAnim.controller
				if activeCtrl == undefined or (classOf fromNodeCtrl) != (classOf activeCtrl) do
				(
					if rObjectMapping.enable_debug then
						format ~FORMAT_OVERWRITTEN~ activeAnim.controller (refs.dependents toNodeCtrl.controller)
					activeAnim.controller = createInstance (classof fromCtrl)
				)
			)
			toCtrl = toNodeCtrl.controller[listI].controller
			
			toCtrl = createInstance (classof fromCtrl)
			toNodeCtrl.available.controller = toCtrl
			toNodeCtrl.controller.setActive toNodeCtrl.controller.count
--			format ~FORMAT_LIST~ listI toNodeCtrl
		)	
		else	
		(		
			-- Assign a new controller to a non-animated parameter, if necessary
--			try(
				if toNodeCtrl.controller == undefined or (classOf fromNodeCtrl) != (classOf toNodeCtrl.controller) do
				(
					if rObjectMapping.enable_debug then
						format ~FORMAT_OVERWRITTEN_TO_NODE_CTRL~ toNodeCtrl.controller (refs.dependents toNodeCtrl.controller)
					toNodeCtrl.controller = createInstance (classof fromNodeCtrl)
				)
--			) catch( if rObjectMapping.enable_debug then format ~MERGE_ERROR_CONTROLLER_CONVERSION~ fromNodeCtrl toNodeCtrl)		
			-- Filter out incompatible controllers (such as Bezier_Position and Position_XYZ)
			if (classOf fromCtrl) != (classOf toNodeCtrl.controller) do return()
			toCtrl = toNodeCtrl.controller
		)
		
		-- Filter out Position/Euler XYZ controllers which return an invalid key array
		if ( (findItem #(Position_XYZ, Euler_XYZ, Master_Point_Controller) (classOf fromCtrl)) == 0) and 
				fromNodeCtrl.keys.count > 0 and toNodeCtrl.controller != fromNodeCtrl do with animate off
		(	
			local ctrlTime, keyTimes = #()
			-- Start building/parsing toCtrl key array
			if cb_matchRange.checked then source_range = (getTimeRange fromCtrl)
			else
			(
				source_range = interval s_startTime.value s_endTime.value
				if source_range.start > (getTimeRange fromCtrl).start do addNewKey fromCtrl source_range.start
				if source_range.end < (getTimeRange fromCtrl).end do addNewKey fromCtrl source_range.end
			)
		if source_range.start < current_range.start do current_range.start = source_range.start
		if source_range.end > current_range.end do current_range.end = source_range.end

			-- Relative offset for fromCtrl values
			if rb_relAbs.state == 1 do
			(
				local toVal = (at time insert_time toCtrl.value)
				local fromVal = (at time source_range.start fromCtrl.value)
				if toVal == undefined then toVal = 0
				if fromVal == undefined then fromVal = 0				
				deltaVal =  toVal - fromVal
--				format "deltaVal:%\n" deltaVal
				ctrlTime = fromCtrl.value
				fromCtrl.value += deltaVal
			)
			-- Offset key times if ~OFFSET_INSERTION_TIME~ is checked
			for ki=1 to fromCtrl.keys.count do
			(
				--local k = if (isKindOf fromCtrl BipSlave_Control) then (biped.getKey fromCtrl ki) else (getKey fromCtrl ki)
				local k = getKey fromCtrl ki
				if (k.time >= source_range.start) and (k.time <= source_range.end) do
				(
					append keyTimes k.time
					k.time += (insert_time - source_range.start)
					append toKeyArray k
				)
			)
			-- Select toCtrl keys to be deleted
			deselectKeys toCtrl
			for k in toCtrl.keys where (k.time > insert_time) and (k.time < (source_range.end + (insert_time - source_range.start))) do selectKeys toCtrl k.time
			deleteKeys toCtrl #selection
			-- Append toKeyArray to toCtrl(destination) key-array
			for ki=1 to toKeyArray.count do
			(
				local k = toKeyArray[ki]
				if (getKeyIndex toCtrl k.time) == 0 do
				(
	/*				if (isKindOf toCtrl BipSlave_Control) then
				(
					local bk = (biped.addNewKey toCtrl k.time)
					for kp in getPropNames k where kp != #selected and kp != #type do
						setProperty bk kp (getProperty k kp)
				)
				else
	*/					appendKey toCtrl.keys k
				)
				k.time = keyTimes[ki] -- restore the original key time				
			)
			sortKeys toCtrl			
			if rb_relAbs.state == 1 do fromCtrl.value = ctrlTime-- restore the original value


		)
		-- Recursively traverse controller sub-anim tree looking for sub-controllers
		for i = 1 to fromNodeCtrl.numsubs do
		(
			--print fromNodeCtrl[i]
			local sub_anim = getSubAnimName fromNodeCtrl i
			--format ~FORMATTED_PASTE_ANIM~ sub_anim fromNodeCtrl[i].controller toCtrl[sub_anim]
			if toCtrl.numsubs >= i and (sub_anim == getSubAnimName toCtrl i) then
				sub_anim = i
			if fromNodeCtrl[i].controller != undefined and toCtrl[sub_anim] != undefined do 
				pasteToActiveCtrl fromNodeCtrl[i].controller &toCtrl[sub_anim]
		)

	)
	
	-- recursiveMergeAnim() recursively merges animation between 2 Animatables.  
	-- Subanims must have matching indices and names to have their animation merged.
	-- recursiveMergeAnim will not step into subAnims that are nodes.
	function recursiveMergeAnim in_fromAnim in_toAnim in_state =
	(
		if in_toAnim != undefined do
		(
			if in_fromAnim.controller != undefined and in_fromAnim.isAnimated do case in_state of
			(
				1:	replaceController in_fromAnim.controller &in_toAnim 
				2:	pasteToActiveCtrl in_fromAnim.controller &in_toAnim
			)
			for i = 1 to in_fromAnim.numSubs do
			(
				local fromSubAnim = in_fromAnim[i]
				if (isKindOf fromSubAnim node) do continue
				local subanim = getSubAnimName in_fromAnim i
				if in_toAnim.numsubs >= i and (subanim == getSubAnimName in_toAnim i) then
					subanim = i
				recursiveMergeAnim fromSubAnim in_toAnim[subanim] in_state
			)
		)
	)
						
	-- Support for animated custom attributes
	function getDefIndex obj str = 
	(
		local defs = custAttributes.getDefs obj
		if defs == undefined then return undefined
		local idx = 1
		for ca in defs do 
		(
			if ca.name == str do return idx
			idx += 1 
		)
		undefined
	)	
	function replaceCAAnim fromAnim toAnim =
	(
		if chkCustAttrib.checked do
		(
			for c = 1 to (custAttributes.count fromAnim) do
			(
				local def = custAttributes.getDef fromAnim c
				local fca = custAttributes.get fromAnim c
				local idx = getDefIndex toAnim def.name
				CAT_CurrentDef = def -- fix for #470704
				
				-- delete the def if it already exists in the dest anim
				--if idx != undefined do custAttributes.delete toAnim idx
				
				-- add the definition from the source anim, and make it unique
				if idx == undefined then
				(
					if chkAddNewDefs.checked then
					(
						if rObjectMapping.enable_debug then
							format ~FORMAT_ADDING_NEW_DEF~ def (custAttributes.getDefSource def)
						custAttributes.add toAnim (execute (custAttributes.getDefSource def))
						idx = (custAttributes.count toAnim)
					)
					else continue;
				)				
				-- now get the dest CA
				local tca = custAttributes.get toAnim idx				
				-- copy the animation
				for j = 1 to fca.numsubs do if j <= tca.numSubs do
				(
					local fromSubAnim = fca[j]; if fromSubAnim.controller == undefined do continue
					local sub_anim = getSubAnimName fca j
					if tca.numsubs >= j and (sub_anim == getSubAnimName tca j) then
						sub_anim = j
					if (tca[sub_anim] != undefined) then
					(
						if fromSubAnim.isAnimated do case rb_ctrl.state of
						(
							1:	replaceController fromSubAnim.controller &tca[sub_anim] 
							2:	pasteToActiveCtrl fromSubAnim.controller &tca[sub_anim]
						)
					)
				)
			)
		)
	)
	-- replaceNodeAnim specifies which node sub-anims are sent to replaceController() and pasteToActiveCtrl()
	function replaceNodeAnim fromNode toNode recurse =
	(
		if (classof fromNode == dotNetMXSValue) then fromNode = fromNode.value
		if rObjectMapping.enable_debug do format "replaceNodeAnim fromNode:% toNode:% recurse:%\n" fromNode toNode recurse
		if fromNode == undefined or toNode == undefined or (isDeleted fromNode) or (isDeleted toNode) then return()
		if recurse do for i=1 to fromNode.children.count do replaceNodeAnim fromNode.children[i] toNode.children[i] recurse
		-- Visibility tracks
		if chkVisTracks.checked and fromNode[1].controller != undefined do
		(
			if toNode[1].controller == undefined then
			(
				animate on (at time 100 toNode.visibility = false)
			)
			case rb_ctrl.state of
			(
				1:	replaceController fromNode[1].controller &toNode[1] 
				2:	pasteToActiveCtrl fromNode[1].controller &toNode[1]
			)
		)
		-- Filter out IK-controlled objects to be dealt downstream
		if (findString (classOf fromNode.controller as string) "IK") == undefined and (findString (classOf toNode.controller as string) "IK") == undefined then
		(
			if chkTransform.checked do
		(
			local chkArray = #(chkPosition.checked, chkRotation.checked, chkScale.checked)
			local toNodeCtrl = toNode.controller
			local tmc = fromNode.controller
			
			if (classOf toNodeCtrl) != (classof tmc) do (toNodeCtrl = toNode.controller = copy tmc)
			if (isKindOf tmc Link_Constraint) do
			(
				tmc = tmc[1];toNodeCtrl = toNodeCtrl[1]
			)
			-- Go through transform sub-anims (PRS)
			if (classof tmc) == prs then
			(
				for j = 1 to 3 do
				(
					if chkArray[j] do case rb_ctrl.state of
					(
						1:	replaceController tmc[j].controller &toNodeCtrl[j]
						2:	pasteToActiveCtrl tmc[j].controller &toNodeCtrl[j]
					)
				)
			)
			else
			(
				for j=1 to tmc.numSubs do
				(
					case rb_ctrl.state of
					(
						1:	replaceController tmc[j].controller &toNodeCtrl[j]
						2:	pasteToActiveCtrl tmc[j].controller &toNodeCtrl[j]
					)
				)
			)
		)
		)
		-- Specific to IK-controlled nodes
		else if chkIK.checked then
		(
			local chkArray = #(chkPosition.checked, chkRotation.checked, chkScale.checked)
			local tmc = fromNode.controller
			-- Pasting from IK goal objects
			if (isKindOf tmc IKChainControl) then
			(
				-- Only paste IK goal's transform controller to non-IK goal object
				if (classof toNode.controller) != (classof tmc) then for j = 1 to 3 do
				(
					if chkArray[j] do case rb_ctrl.state of
					(
						1:	replaceController tmc[2][j].controller &toNode.controller[j]
						2:	pasteToActiveCtrl tmc[2][j].controller &toNode.controller[j]
					)
				)
				-- IK goal to IK goal paste
				else
				(

					--IK goal Swivel param
					if tmc[1].controller != undefined do case rb_ctrl.state of
					(
						1: 	replaceController tmc[1].controller &toNode.controller[1]
						2:	pasteToActiveCtrl tmc[1].controller &toNode.controller[1]
					)
					--IK goal transforms
					for j = 1 to 3 do
					(
						if chkArray[j] and tmc[2][j].isAnimated do case rb_ctrl.state of
						(
							1:	replaceController tmc[2][j].controller &toNode.controller[2][j] 
							2:	pasteToActiveCtrl tmc[2][j].controller &toNode.controller[2][j]
						)
					)	
					if (isKindof toNode.controller[3].controller On_Off) then
					(
					--Hack to paste IK Enabled keys
					deleteKeys toNode.controller[3].keys #allKeys
					for k in tmc[3].keys do addNewKey toNode.controller[3].keys k.time
					if rb_ctrl.state == 2 do
					(
						local 	tempCtrl = toNode.controller[3].controller,
								source_range = interval s_startTime.value s_endTime.value,
								insert_time  = s_insert.value
						if not cb_matchRange.checked do setTimeRange tempCtrl source_range
						insertTime tempCtrl (getTimeRange tempCtrl).start (insert_time - (getTimeRange tempCtrl).start)
					)			
				)
					else -- maybe the new boolean controller
					(
						case rb_ctrl.state of
						(
							1:	replaceController tmc[3].controller &toNode.controller[3]
							2:	pasteToActiveCtrl tmc[3].controller &toNode.controller[3]
						)
			)
				)
			)
			-- Pasting from IK bone objects
			else if (isKindOf tmc IKControl) then
			(
				-- Only paste IK bone's FK_Sub_Contrl to non-IK object
				if (classof toNode.controller) != (classof tmc) then for j = 1 to 3 do
				(
					if chkArray[j] do case rb_ctrl.state of
					(
						1:	replaceController tmc[4][j].controller &toNode.controller[j]
						2:	pasteToActiveCtrl tmc[4][j].controller &toNode.controller[j]
					)
				)
				-- Bone to bone pasting				
				else 
				(
					-- Bone to bone pasting				
					for j = 1 to 3 do -- for preferred angles
					(
						if chkArray[j] and tmc[j].controller != undefined do case rb_ctrl.state of
						(
							1:	replaceController tmc[j].controller &toNode.controller[j]
							2:	pasteToActiveCtrl tmc[j].controller &toNode.controller[j]
						)
					)
					for j = 1 to 3 do -- fk sub control
					(
						if chkArray[j] do case rb_ctrl.state of
						(
							1:	replaceController tmc[4][j].controller &toNode.controller[4][j]
							2:	pasteToActiveCtrl tmc[4][j].controller &toNode.controller[4][j]
						)
					)
				)
			)
			else -- maybe a spline IK control
			(
				--print ~I_AM_IK_SPLINE~
				for j=1 to tmc.numSubs do
				(
					case rb_ctrl.state of
					(
						1:	replaceController tmc[j].controller &toNode.controller[j]
						2:	pasteToActiveCtrl tmc[j].controller &toNode.controller[j]
					)
				)
			)			
		)		
		-- Support for animated base object sub-anims based on sub-anim name
		if chkBaseObject.checked do
		(
			local from_baseObject = fromNode.baseObject
			local to_baseObject = toNode.baseObject
			for j = 1 to from_baseObject.numsubs do
			(
				local fromSubAnim = from_baseObject[j]; if fromSubAnim.controller == undefined do continue
				local sub_anim = getSubAnimName from_baseObject j
				if to_baseObject.numsubs >= j and (sub_anim == getSubAnimName to_baseObject j) then
					sub_anim = j
				if to_baseObject[sub_anim] != undefined and fromSubAnim.isAnimated do case rb_ctrl.state of
				(
					1:	replaceController fromSubAnim.controller &to_baseObject[sub_anim] 
					2:	pasteToActiveCtrl fromSubAnim.controller &to_baseObject[sub_anim]
				)
			)			
		)
		replaceCAAnim fromNode.baseobject toNode.baseobject
		-- Support for animated modifier sub-anims based on modifier name
		if chkModifiers.checked and toNode.modifiers.count > 0 do
		(
			for m=1 to fromNode.modifiers.count do 
			(
				local from_mod = fromNode.modifiers[m]
				local to_mod = toNode.modifiers[m]
--				format "%-%-%\n" m (classof from_mod) (classof to_mod)
				if (classof from_mod) == (classof to_mod) then
				(
					recursiveMergeAnim from_mod to_mod rb_ctrl.state
					replaceCAAnim from_mod to_mod
				)
			)
		)
		if chkMaterials.checked and fromNode.material != undefined then
		(
			local mat = fromNode.material
			if toNode.material == undefined then
				toNode.material = createInstance (classof mat)
			if (classof toNode.material) == (classof mat) do
			(
				for j = 1 to mat.numsubs do
				(
					recursiveMergeAnim mat[j] toNode.material[j] rb_ctrl.state
					replaceCAAnim fromNode.material toNode.material
				)
			)
		)	
	)
	function replaceAnim fromAnim toAnim recurse =
	(
		if fromAnim.numSubs != toAnim.numSubs do return false
		if fromAnim.numSubs == 0 then
		(
			try
			(
				toAnim.controller = fromAnim.controller
			)
			catch
			( 
				SetStatus (~COULD_NOT_SET_ANIMATION~ + toAnim as string + " < - " + fromAnim as string) error:true
			)
		)
		for j=1 to fromAnim.numSubs do
		(
			if fromAnim[j].isAnimated then
				toAnim[j].controller = fromAnim[j].controller
		)
	)	
	
	-- these functions are used by the character script
	function resetAnim fromAnim =
	(
		--format ~RESET_ANIM_CONTROLLER~ (fromAnim.controller as string)
		if fromAnim.controller != undefined then
			deleteKeys fromAnim.controller #allKeys			
		
		-- delete anims from custom attributes
		--format "\tReset Anim CA.count: %\n" (custAttributes.count fromAnim) 
		for k=1 to (custAttributes.count fromAnim) do 
		(
			local ca = custattributes.get fromAnim k
			--format "\tReset Anim CA[%]: %\n" k (ca as string) 
			if (ca != undefined) do
			(
				local saNames = getSubAnimNames ca
				--format "\t\tReset Anim CA.subAnims.Count: %\n" saNames.count 
				for s=1 to saNames.count do 
				(
					--format "\t\tReset Anim CA.subAnims[%]: %\n" s ca[s] 
					if (ca[s].controller != undefined) do (
						--format "\t\t\tReset Anim CA.subAnims[%].controller: %\n" s (ca[s].controller as string) 
						deleteKeys ca[s].controller #allKeys
					)	
				)
			)
		)

		--format "\tReset Anim SubAnims.count: %\n" fromAnim.numSubs 
		for j=1 to fromAnim.numSubs do
		(
			resetAnim fromAnim[j]
		)
	)
	function resetNodeAnim node = 
	(
		--format "Reset Anim Node: %\n" node.name
		--format "Reset Anim Node.Controller: %\n" (node.controller as string)
		resetAnim node.controller
		--format "Reset Anim Node.BaseObject: %\n" (node.baseobject as string)
		resetAnim node.baseObject
		for m in node.modifiers do (
		--format "Reset Anim Node.modifier: %\n" (m as string)
			resetAnim m
		)
	)
	-------------------------------------------------------------------------------------------
	-- File I/O Stuff
	-------------------------------------------------------------------------------------------	
	-- loadFileInfo() has been modified to merge source objects into a group; this is somewhat faster than using XRefs and provides more direct access to source nodes
	--	function unloadXRef = ( try (delete new_xref) catch () )
--	function unloadXRef = ( try (delete new_tree) catch () )
	function unloadXRef = 
	( 
		try 
		(
			xml_input = false
			if new_xref != undefined then delete new_xref
			if $_merge_anim != undefined do delete $_merge_anim*			
		) catch() 
	)
	
	function loadXMLFile fname = 
	(
		if fname == undefined then return false
		setWaitCursor()
		xml_input = true		
		new_xref = undefined
		file_name = fname
		--format "getTextExtent: %\n" (getTextExtent (getFileNamePath file_name))
		local location = ~LOCATION_CAPTION~
		local index = location.count+1
		location +=  getFileNamePath file_name
		local maxLen = 377
		while (getTextExtent location).x > maxLen and index <= location.count do (location[index]=".";index += 1)
		--format "getTextExtent: %\n" (getTextExtent (getFileNamePath file_name))
		lblProps.caption =
			~FILE_CAPTION~ + (getFileNameFile file_name) + ".xml" +
			"\n" + location

		xmlIO.init()
		if (xmlIO.load fname) then
		(
			new_tree = xmlIO.xmlDoc.selectSingleNode("//objects")
			source_nodes = new_tree.childNodes
			lvops.ClearLvItems rObjectMapping.lvSource
			addXMLNodesToListView (rObjectMapping.lvSource) source_nodes recurse:true
		)
		else
		(
			local reason = xmlIO.xmlDoc.parseError.reason as string
			local srcText = xmlIO.xmlDoc.parseError.srcText as string
			setStatus (reason  + " at " + srcText) error:true
		)
		setArrowCursor()
		true
	)
	function loadFileInfo fname =
	(
		if fname == undefined then return false		
		setWaitCursor()		
		new_xref = undefined
		file_name = fname

		rMergeAnim.unloadXRef()

		-- Create new group at world origin to ensure a 1-1 correspondence with source and destination node transformation
		new_tree = group (local temp_node = point()) name:"_merge_anim"
		local obj_count = objects.count
		local res = mergeMAXFile file_name #mergedups #noRedraw
		if not res then return false
		
		-- Collect all newly merged objects into a new temp group
		local gobjs = for i = (obj_count + 1) to objects.count collect
				( objects[i].isHidden = true; objects[i] )
		local temp_group = group gobjs
		
		if temp_group != undefined and temp_group != ok then -- group created
		(
			temp_group.isHidden = true
			
			-- Transfer objects from the temp group into new_tree
			for c in temp_group.children do c.parent = new_tree; delete #(temp_node, temp_group)
		
			new_tree.isHidden = true
			new_tree_count = new_tree.children.count
			clearUndoBuffer()
		)
		else -- bring in the source file as an xref
		(
			for i = gobjs.count to 1 by -1 do ( try (if not (isDeleted gobjs[i]) then delete gobjs[i]) catch(format ~ERROR_DELETING_TEMP_NODE~ gobjs[i]) )
			if file_name == (maxFilePath + maxFileName) then
			( 				MessageBox ~MSGBOX_CANNOT_OPEN_THE_FILE_AS_SRC~
				return false
			)
			xrefs.addNewXrefFile file_name
			new_xref = xrefs.getXRefFile (xrefs.getXRefFileCount())
			new_xref.hidden = true
			new_tree = new_xref.tree						
		)
		local location = ~LOCAL_LOCATION_CAPTION~
		local index = location.count+1
		location +=  getFileNamePath file_name
		local maxLen = 377
		while (getTextExtent location).x > maxLen and index <= location.count do (location[index]=".";index += 1)
		--format "getTextExtent: %\n" (getTextExtent (getFileNamePath file_name))
		lblProps.caption =
			~FILE_GETFILENAMEFILE_CAPTION~ + (getFileNameFile file_name) + (getFileNameType file_name) +
			"\n" + location

		setArrowCursor()
		true
	)	

	function loadSource f =
	(
		setWaitCursor()
		local shortFileName = getFileNameFile f + getFileNameType f
		--format "loadSource - shortFileName: %\n" shortFileName 
		if not (doesFileExist shortFileName) do
		(
			SetStatus (~SETSTATUS_FAILED_TO_LOAD_FILE~ + shortFileName) error:true
			return false
		)

		if $_merge_anim_old != undefined do delete $_merge_anim_old*
		if $_merge_anim != undefined do $_merge_anim.name = "_merge_anim_old"

		unLoadXRef()
		setStatus (~LOADING_SOURCE_FILE~ + shortFileName)
		local okLoad = loadFileInfo f
		if okLoad then
		(
			lvops.ClearLvItems rObjectMapping.lvSource

			addSceneObjectsToListView (rObjectMapping.lvSource) (source_nodes = new_tree.children) recurse:true
			setStatus ~LOADED_SOURCE_FILE_SUCCESSFULLY~
			if $_merge_anim_old != undefined do delete $_merge_anim_old*
		)
		else 
		(
			SetStatus (~SETSTATUS_FAILED_TO_LOAD_FILE_SHORTFILENAME~ + shortFileName) error:true
			if $_merge_anim_old != undefined do $_merge_anim_old.name = "_merge_anim"
		)
		setArrowCursor()
		okLoad 
	)
	function saveMapping =
	(
		setWaitCursor()
		format "\"%\" %\n" (if file_name == undefined then "" else file_name) rObjectMapping.lvCurrent.items.count to:map_stream
		local LvItems = lvops.GetLvItems rObjectMapping.lvCurrent
		for li in LvItems do
		(
			format "\"%\" % % %" li.text (li.forecolor.toARGB()) li.font.bold li.tag.value.handle to:map_stream
			
			local si = li.subitems.item[merge_col]
			if si.tag == undefined or si.text == "" then
			(
				format " \"\" \"\"\n" to:map_stream
			)
			else
			(
				--the tag property holds a DotNetMXSValue whose .value property holds a mxs Value object
				local tag = li.SubItems.item[merge_col].tag.value
				format " \"%\" %\n" si.text (if xml_input then tag else tag.handle ) to:map_stream
			)
		)
		setArrowCursor()
	)
	function loadMapping =
	(
		--format "loadMapping - source_nodes: %\n" source_nodes 
		local items = lvops.GetLvItems rObjectMapping.lvCurrent
		local num_lines, ln=0, src_file 
		seek map_stream 0
		src_file = readValue map_stream ignoreStringEscapes:true -- scene name
		--format "loadMapping - src_file: %;\n file_name: %\n" src_file file_name 
		
		-- if a source file is listed then see if it is different from the existing one and load it
		if src_file != "" and src_file != file_name then 
		( 
			local res = loadSource src_file
			if not res then return false -- source file not found. Bail!
			lvops.RefreshListView rObjectMapping.lvSource 
		)
		--format "loadMapping - source_nodes: %\n" source_nodes 
		
		num_lines = readValue map_stream
		
		setWaitCursor()
		SetStatus ~LOADING_MAPPING_FILE~
		lvops.ClearLvItems rObjectMapping.lvCurrent

		while not eof map_stream do
		(
			local li, li_tag, si, si_tag, li_node
/*>>*/		local text = readValue map_stream ignoreStringEscapes:true
			li = lvops.AddLvItem rObjectMapping.lvCurrent pTextItems: #(text,"")

			--Read in two items from the stream, however they are not used.
/*>>*/		local unusedOLEColor = readValue map_stream
/*>>*/		local unusedFontStyle = readValue map_stream

			--Read in the scene object handle value, and get a maxscript handle to it.
/*>>*/		li_node = getNodeByHandle (readValue map_stream)
			--Read in the scene object string name 
/*>>*/		si = readValue map_stream ignoreStringEscapes:true -- target node name
			--Read in the scene object handle value 
/*>>*/		si_tag = readValue map_stream
			
			--format "loadMapping - li.text: %; li.forecolor: %; li_node: %; si: %; si_tag: %\n" \
			--	li.text (li.forecolor.toARGB()) li_node.name si si_tag
			
			li_tag = if li_node == undefined then execute ("$'" + li.text + "'") else li_node
				
			--Properly highlight the text of the scene node names	
			local isAnimated = xmlio.isAnimated li_tag
			if (hasProcedural li_tag) then
			(
				li.UseItemStyleForSubItems = false
				local subItem = li.subitems.item[0]
				lvops.HighLightLvItem subItem #bold proc_color
			)
			else if isAnimated and not rObjectMapping.chkOnlyAnim.checked do
			(
				li.indentCount = rObjectMapping.spnIndent.value
				li.UseItemStyleForSubItems = false
				local subItem = li.subitems.item[0]
				lvops.HighLightLvItem subItem #bold anim_color
			)
				
			li.tag = dotnetMXSValue li_tag
			if li_tag == undefined then 
			( 
				li.UseItemStyleForSubItems = false
				lvops.HighLightLvItem li.subitems.item[0] #bold err_color 
			)
			--format "loadMapping - li_tag: %\n" li_tag
			if si != "" then
			(
				li.subItems.item[merge_col].text = si
				si_node = getNodeByHandle si_tag
				local dnTag = dotNetMXSValue (if si_node == undefined then execute ("$'" + si + "'") else si_node)
				li.SubItems.item[merge_col].tag = dnTag 
				if si_node == undefined then 
				(
					li.UseItemStyleForSubItems = false
					lvops.HighLightLvItem li.SubItems.item[0] #regular err_color
				)

				--format "loadMapping - si_node: %; li.SubItems.item[merge_col].tag: %\n" si_node li.subItems.item[merge_col].tag
			)
			ln += 1
			pbStatus.value = 100.*ln/num_lines
			if (mod pbStatus.value 10) == 0 then	gc() -- for every 10%, do a gc
		)
		pbStatus.value = 0
		setArrowCursor()
		--format "loadMapping - source_nodes: %\n" source_nodes 
		true
	)
	
	function Save2XML filename nodes: =
	(
		if filename != undefined then
		(
			setWaitCursor()
			savedNodes = 0
			if nodes == unsupplied then nodes = rootNode.children
			obj_count = nodes.count			
			setStatus ~SAVING_XML_FILE~
			xmlIO.init()
			local objsElem = xmlIO.xmlDoc.createElement "objects"
			xmlIO.world.appendChild objsElem
			for o in nodes do
			(
				xmlIO.obj2xml o objsElem  postCallback:(rMergeAnim.postSaveCB)
			)
			xmlIO.save filename
--			messageBox (xmlIO.no_key_frames as string)
			setStatus ~XML_FILE_SAVED_SUCCESSFULLY~
			if show_messages then
				MessageBox ~MSGBOX_SAVE_COMPLETED~ title:~MSGBOX_SAVE_TITLE~ beep:true
			setArrowCursor()
		)
	)

	function RegisterCallbacks =
	(
		callbacks.addScript #systemPreReset	"destroyDialog rMergeAnim" 	id:#rMergeAnim
		callbacks.addScript #systemPreNew		"destroyDialog rMergeAnim"	id:#rMergeAnim
		callbacks.addScript #filePreOpen		"destroyDialog rMergeAnim" 	id:#rMergeAnim
	
--		callbacks.addScript #filePreSave		"rMergeAnim.unloadXRef()" 	id:#rMergeAnim
--		callbacks.addScript #filePostSave		"rMergeAnim.updateTrees()" id:#rMergeAnim

	)
	-------------------------------------------------------------------------------------------
	-- Initialization Stuff
	-------------------------------------------------------------------------------------------
	on rMergeAnim open do
	(
		replaceController_fn = replaceController
		
		setWaitCursor()
		xmlio = gxmlIO
		file_name = undefined
		dontFilterChildren = false
		addSubRollout srRollouts rObjectMapping	rolledUp:false
			
		-- add other procedural types
		append proc_types position_reactor
		append proc_types rotation_reactor
		append proc_types scale_reactor
		append proc_types float_reactor		
		append proc_types BipSlave_Control
		append proc_types Footsteps
		append proc_types Biped_SubAnim
		
		-- Initialize various controls
		initStatusBar()
		
		btnSave2XML.visible = rObjectMapping.enable_debug
		
		if fname != undefined and (loadFileInfo fname) then
		(
			local start, end 
			start = timeStamp()
			
			addSceneObjectsToListView (rObjectMapping.lvSource) (source_nodes = new_tree.children) recurse:true
			
			end = timeStamp()
			format ~FORMAT_TOTAL~ ((end-start)/1000)
		)
		local baby = "haydon"
		
		s_insert.enabled = false
		current_nodes = rootNode.children
		
		addSceneObjectsToListView rObjectMapping.lvCurrent current_nodes
		
		callbacks.removeScripts id:#rMergeAnim
		RegisterCallbacks()
		
		setArrowCursor()
		rObjectMapping.doRollup true
	)
	on rMergeAnim close do
	(
		setIniSetting ini_file #general #position ((getDialogpos rMergeAnim) as string)
		callbacks.removeScripts id:#rMergeAnim
		unloadXRef()
	)
	-------------------------------------------------------------------------------------------
	-- UI options events
	-------------------------------------------------------------------------------------------
	on chkTransform		changed state do ( chkPosition.enabled = chkRotation.enabled = chkScale.enabled = state)
	on chkCustAttrib	changed state do chkAddNewDefs.enabled = state

	on cb_matchRange changed state do s_startTime.enabled = s_endTime.enabled = not state
	on rb_ctrl changed state do case state of
	(
		1:	s_startTime.enabled = s_endTime.enabled = cb_matchRange.enabled = s_insert.enabled = rb_relAbs.enabled = false

		2:	(
				s_startTime.enabled = s_endTime.enabled = (not cb_matchRange.state)
				cb_matchRange.enabled = true
				s_insert.enabled = rb_relAbs.enabled = true
			)
	)
	
	on btnFetch pressed do
	( 
		callbacks.removeScripts id:#rMergeAnim		
		setStatus ~FETCHING_IN_PROGRESS~
		if is_holding then
		(
			try
			(
				unloadXRef()
				delete hold_objects							
				mergeMaxFile hold_file --#deleteOldDups				
			) catch()
--			map_stream = stringStream ""; loadMapping()
		)
		RegisterCallbacks()		
		if (getFileNameType file_name) == ".xml" then loadXMLFile file_name
		else 
		( 
			loadSource file_name; 
			if (new_tree != undefined) then
			(
				new_tree.isHidden = true 
			)
		)
		updateSource()
		updateCurrent()		
		setStatus ~FETCHING_DONE~
		btnFetch.enabled = false
	)
	-------------------------------------------------------------------------------------------
	-- Do the animation merging/replacing
	-------------------------------------------------------------------------------------------
	on btnSource pressed do
	(
		local f = getOpenFileName filename:sourceSeed caption:~SOURCE_FILE_CAPTION~ types:~MAX_ANIMATION_FILE_TYPES~
		if f != undefined do
		(
		    sourceSeed = f
			if (getFileNameType f) == ".xml" then loadXMLFile f
			else loadSource f
			updateCurrent()
		)
 	)
	on btnSourceObj pressed do
	(
		if hitByNameDlg() then 
		(
			unloadXref()
			file_name = ""			
			source_nodes = selection

			setStatus ~LOADING_SOURCE_OBJECTS~
			lvops.ClearLvItems rObjectMapping.lvSource
		
			addSceneObjectsToListView (rObjectMapping.lvSource) source_nodes recurse:false
			updateCurrent()
			setStatus ~SOURCE_OBJECTS_SUCCESSFULLY~
		)
 	)	
	on btnSave2XML pressed do
	(
		local f = getSaveFileName types:~XML_SCENE_FILE_TYPES~
		Save2XML f
	)
	on btnMerge pressed do
	(
		local st = timeStamp(), dt, et=0, ot=0, ct=0, merged_objects=0
		
		local items      = lvops.GetLvItems rObjectMapping.lvCurrent
		local item_count = items.count
		if item_count == 0 then
		(
			if show_messages then MessageBox ~MSGBOX_NO_ITEMS_TO_MERGE~ title:~MERGE_ANIMATION~ beep:true
			return()
		)
					
		is_holding = true		
		deleteFile hold_file
		btnFetch.enabled = true
		hold_objects = #()
		-- save all nodes to a temp file to provide undoing the merge
		saveNodes objects hold_file
		gc light:true
		
		map_stream = stringStream ""--; saveMapping()
		setWaitCursor()
		setStatus ~MERGING_ANIMATION~
		xmlio.animatedOnly = false		
		
		with redraw Off
		(
            -- run through all the list items and do the merge
			local i = 1
			for li in items do
			(		
				local si = li.SubItems.item[merge_col]
				if si != undefined then
				(
--					try
--					(
						local fromNode = undefined
						local toNode = li.tag.value
						if toNode == undefined or (isDeleted toNode) do continue
						if xml_input then
						(
							local fromElem = xmlIO.xmlDoc.selectSingleNode ("//object[@id='" + si.tag + "']")							 

							local ots = timeStamp()
							
							-- create the object from the xml element
							fromNode = xmlIO.xml2obj	fromElem chkTransform:chkTransform.checked \
														chkBaseObject:chkBaseObject.checked \
														chkModifiers:chkModifiers.checked \
														chkCustAttrib:chkCustAttrib.checked
							ot += timeStamp() - ots
							local cls = (xmlIO.getAttribute fromElem #classOf)
							if fromNode != undefined then
							(
								replaceNodeAnim fromNode toNode false
								delete fromNode
--								gc()
							)
							else
								format ~FORMAT_CANNOT_CREATE_NODE~ (xmlIO.getAttribute fromElem #name) cls
						)
						else
						(
							local skip = false							
							--try (fromNode = si.tag.value) catch (skip = true)
							if (si.tag != undefined) then
							(	
								fromNode = si.tag.value
								if not skip and (getNodeByName si.text) != undefined then 
								(
									append hold_objects toNode
									replaceNodeAnim fromNode toNode false
								)
								else 
								(
									li.UseItemStyleForSubItems = false
									lvops.HighLightLvItem (li.subitems.item[0]) #bold err_color
								)
							)
							else
							(
								skip = true
							)
						)
						
						if xmlio.isAnimated toNode do 
						( 
							li.UseItemStyleForSubItems = false
							lvops.HighLightLvItem (li.subitems.item[0]) #bold anim_color
						)
						merged_objects += 1
--					)
/*					catch
					(
						-- incase of an error, highlight the list item
						li.UseItemStyleForSubItems = false
						lvops.HighLightLvItem li #bold err_color
						format ~FORMATTED_ERROR_OCCURED_IN_PROCESSING_LIST_NODE~ li.text					
						throw()
					)
*/				
				)
				pbStatus.value = 100.*i/item_count
				i += 1
				if (mod pbStatus.value 10) == 0 then gc() -- for every 10%, do a gc
			) -- end for loop
		)
		setArrowCursor()

		-- Timings
		dt = timeStamp() - st
		if xml_input then
		(
			et = xmlio.exec_time
			ct = xmlio.create_time
		)
		
		if rObjectMapping.enable_debug then
		(
			format ~FORMATTED_TOTAL_TIME~  (dt/1000)
			format ~FORMATTED_EXECUTE_TIME~ (et*100/dt)
			format ~FORMATTED_OBJECT_CREATION_TIME~ (ot*100/dt)
			format ~FORMATTED_CREATEINSTANCE_TIME~ (ct*100/dt)
		)
		if chkAdjustTimeRange.checked and current_range.start != current_range.end do animationRange = current_range

		if show_messages then MessageBox (if merged_objects == 0 then ~MERGE_NODES_COLUMN_IS_EMPTY_NO_OBJECTS_MERGED~  else ~MERGE_COMPLETED~) title:~MERGE_ANIMATION_TITLE~ beep:true	
		setStatus (~TOTAL_MERGED~ + merged_objects as string)
		pbStatus.value = 0
	)
	function postSaveCB obj = 
	(
		savedNodes += 1
		pbStatus.value = savedNodes*100/obj_count
	)
	
	function openDialog = 
	(
		local pos = execute (getIniSetting ini_file #general #position)
		--local rolledUp = (execute (getIniSetting ini_file #general #rolledUp)) == true
		if pos == ok then pos = [100, 100]
		createDialog rMergeAnim pos:pos style:#(#style_border, #style_titlebar, #style_minimizebox, #style_maximizebox, #style_sysmenu) escapeEnable:false
	)	
  ) --end of rollout

--rMergeAnim.openDialog()


-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQBfhpUfy9kCw3PUChsZKg62HO9lkO8JjQp7
-- 8BYm0lzrm7tPQMJbBudGZqLj3USaFmv44rWM42cjAKPzKQIYCOiHmEKuCUcgQ6lz
-- QEYRXctsXRjtey+jJ9UnWU2jbAZd57eMD9vMY+MriQAtV7rHAS0DYZw6OCM3hZf6
-- KHVLKB5Y2oNVnfm0Uhb5ApXo7vB/36YO13uNSRv1fqVQKFXl0mpn573S8Nq6jW5w
-- asRbMOkSZPNa5j/Pqh/Q3jPJfZGGwc2AzVXtngQu8yQX8bq6dKwBK1LhabMREj+4
-- mAi0wEBjtp0qtKDwRDtF1VrEQwvrJEozIEC8J//3o+6YmBy53uTg
-- -----END-SIGNATURE-----